<?php
// Voter Login Page
require_once '../includes/db.php';

// If logged in already, send voter to correct page
if (isset($_SESSION['voter_id']) && !empty($_SESSION['voter_id'])) {

    // Check if they have voted before redirect
    $sql = "SELECT has_voted FROM voters WHERE id = ?";
    if ($stmt = $conn->prepare($sql)) {
        $stmt->bind_param("i", $_SESSION['voter_id']);
        $stmt->execute();
        $stmt->bind_result($has_voted);
        $stmt->fetch();
        $stmt->close();

        if ($has_voted == 1) {
            header("location: view_votes.php"); // <-- PAGE THAT SHOWS ONLY THEIR RESULT
            exit;
        } else {
            header("location: vote.php"); // <-- NORMAL VOTING PAGE
            exit;
        }
    }
}

$staff_id = $password = "";
$staff_id_err = $password_err = $login_err = "";

// Handle login submission
if ($_SERVER["REQUEST_METHOD"] == "POST") {

    // Staff ID
    if (empty(trim($_POST["staff_id"]))) {
        $staff_id_err = "Please enter Staff ID.";
    } else {
        $staff_id = trim($_POST["staff_id"]);
    }

    // Password
    if (empty(trim($_POST["password"]))) {
        $password_err = "Please enter your password.";
    } else {
        $password = trim($_POST["password"]);
    }

    // Validate entries
    if (empty($staff_id_err) && empty($password_err)) {

        $sql = "SELECT id, staff_id, full_name, password, has_voted 
                FROM voters WHERE staff_id = ?";

        if ($stmt = $conn->prepare($sql)) {

            $stmt->bind_param("s", $param_staff_id);
            $param_staff_id = $staff_id;

            if ($stmt->execute()) {

                $stmt->store_result();

                if ($stmt->num_rows == 1) {

                    $stmt->bind_result($id, $staff_id, $full_name, $hashed_password, $has_voted);
                    $stmt->fetch();

                    if (password_verify($password, $hashed_password)) {

                        // Login successful
                        session_regenerate_id();
                        $_SESSION["voter_id"] = $id;
                        $_SESSION["voter_name"] = $full_name;
                        $_SESSION["voter_staff_id"] = $staff_id;

                        // Redirect based on voting status
                        if ($has_voted == 1) {
                            header("location: view_votes.php");  // <-- CANNOT VOTE AGAIN
                            exit;
                        } else {
                            header("location: vote.php"); // <-- CAN VOTE
                            exit;
                        }

                    } else {
                        $login_err = "Invalid Staff ID or password.";
                    }

                } else {
                    $login_err = "Invalid Staff ID or password.";
                }

            } else {
                echo "Oops! Something went wrong. Please try again later.";
            }

            $stmt->close();
        }
    }
}

require_once '../includes/header.php';
?>

<h4 class="font-weight-light">Voter Login</h4>
<h6 class="font-weight-light">Sign in with your Staff ID.</h6>

<?php
if (!empty($login_err)) {
    echo '<div class="alert alert-danger">' . $login_err . '</div>';
}
?>

<form class="pt-3" action="<?php echo htmlspecialchars($_SERVER["PHP_SELF"]); ?>" method="post">
    <div class="form-group">
        <input type="text" name="staff_id" class="form-control form-control-lg <?php echo (!empty($staff_id_err)) ? 'is-invalid' : ''; ?>" placeholder="Staff ID" value="<?php echo $staff_id; ?>">
        <span class="invalid-feedback"><?php echo $staff_id_err; ?></span>
    </div>
    <div class="form-group">
        <input type="password" name="password" class="form-control form-control-lg <?php echo (!empty($password_err)) ? 'is-invalid' : ''; ?>" placeholder="Password">
        <span class="invalid-feedback"><?php echo $password_err; ?></span>
    </div>
    <div class="mt-3">
        <button type="submit" class="btn btn-block btn-gradient-primary btn-lg font-weight-medium auth-form-btn">LOGIN</button>
    </div>
</form>

<?php require_once '../includes/footer.php'; ?>
