<?php
// View Voters Management
require_once '../../includes/header.php';
require_once '../../includes/email_mock.php';
check_admin_login();

$message = '';

/* ------------------------------
    HELPER: Generate Random Password
------------------------------- */
function generate_random_password($length = 10) {
    $chars = 'abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789!@#$%^&*()';
    $password = '';
    for ($i = 0; $i < $length; $i++) {
        $password .= $chars[rand(0, strlen($chars) - 1)];
    }
    return $password;
}

/* ------------------------------
    ADD NEW VOTER (Modal Form)
------------------------------- */
if (isset($_POST["add_voter"])) {

    $staff_id   = trim($_POST["staff_id"]);
    $full_name  = trim($_POST["full_name"]);
    $email  = trim($_POST["email"]);
    $department = trim($_POST["department"]);

    if (empty($staff_id) || empty($full_name) || empty($email) || empty($department)) {
        $message = '<div class="alert alert-danger">All fields are required.</div>';

    } else {

        $raw_password   = generate_random_password(10);
        $hashed_password = password_hash($raw_password, PASSWORD_BCRYPT);

        $sql = "INSERT INTO voters (staff_id, full_name, email, department, password) 
                VALUES (?, ?, ?, ?, ?)";

        if ($stmt = $conn->prepare($sql)) {

            $stmt->bind_param("sssss", $staff_id, $full_name, $email, $department, $hashed_password);

            if ($stmt->execute()) {
                send_voter_credentials_email($staff_id, $raw_password, $full_name);

                $msg = "Voter added successfully. Password logged in email log.";
                $message = '<div class="alert alert-success">'.$msg.'</div>';

            } else {
                if ($conn->errno == 1062) {
                    $message = '<div class="alert alert-danger">Staff ID already exists.</div>';
                } else {
                    $message = '<div class="alert alert-danger">Error: '.$conn->error.'</div>';
                }
            }
            $stmt->close();
        }
    }
}

/* ------------------------------
    UPDATE VOTER
------------------------------- */
if (isset($_POST["edit_voter"])) {

    $id         = $_POST["voter_id"];
    $staff_id   = trim($_POST["staff_id"]);
    $full_name  = trim($_POST["full_name"]);
    $full_name  = trim($_POST["email"]);
    $department = trim($_POST["department"]);

    if (!empty($staff_id) && !empty($full_name) && !empty($department)) {

        $sql = "UPDATE voters SET staff_id=?, full_name=?, email=?, department=? WHERE id=?";
        if ($stmt = $conn->prepare($sql)) {

            $stmt->bind_param("ssssi", $staff_id, $full_name, $email, $department, $id);

            if ($stmt->execute()) {
                $message = '<div class="alert alert-success">Voter updated successfully.</div>';
            } else {
                if ($conn->errno == 1062) {
                    $message = '<div class="alert alert-danger">Staff ID already exists!</div>';
                } else {
                    $message = '<div class="alert alert-danger">Error: '.$conn->error.'</div>';
                }
            }
            $stmt->close();
        }

    } else {
        $message = '<div class="alert alert-warning">Invalid data provided.</div>';
    }
}

/* ------------------------------
    DELETE VOTER
------------------------------- */
if (isset($_GET["action"]) && $_GET["action"] == "delete" && isset($_GET["id"])) {
    $id = intval($_GET["id"]);

    $sql = "DELETE FROM voters WHERE id=?";
    if ($stmt = $conn->prepare($sql)) {
        $stmt->bind_param("i", $id);
        if ($stmt->execute()) {
            $message = '<div class="alert alert-success">Voter deleted successfully.</div>';
        } else {
            $message = '<div class="alert alert-danger">Error: '.$conn->error.'</div>';
        }
        $stmt->close();
    }
}


/* ------------------------------
    CANCEL VOTE FOR SELECTED VOTER
------------------------------- */
if (isset($_GET["action"]) && $_GET["action"] == "cancel_vote" && isset($_GET["id"])) {
    $voter_id = intval($_GET["id"]);

    $conn->begin_transaction();
    $ok = true;

    // Step 1 — Delete votes cast by this voter
    $sql_del = "DELETE FROM votes WHERE voter_id = ?";
    if ($stmt1 = $conn->prepare($sql_del)) {
        $stmt1->bind_param("i", $voter_id);
        if (!$stmt1->execute()) { $ok = false; }
        $stmt1->close();
    } else { $ok = false; }

    // Step 2 — Reset has_voted
    $sql_reset = "UPDATE voters SET has_voted = 0 WHERE id = ?";
    if ($stmt2 = $conn->prepare($sql_reset)) {
        $stmt2->bind_param("i", $voter_id);
        if (!$stmt2->execute()) { $ok = false; }
        $stmt2->close();
    } else { $ok = false; }

    if ($ok) {
        $conn->commit();
        $message = '<div class="alert alert-success">Vote cancelled successfully. Voter can vote again.</div>';
    } else {
        $conn->rollback();
        $message = '<div class="alert alert-danger">Error cancelling vote. Please try again.</div>';
    }
}




/* ------------------------------
    FETCH ALL VOTERS
------------------------------- */
$voters = [];
$sql = "SELECT id, staff_id, full_name, email, department, has_voted, created_at 
        FROM voters ORDER BY id DESC";
$res = $conn->query($sql);
while ($row = $res->fetch_assoc()) {
    $voters[] = $row;
}
?>

<!-- PAGE HEADER -->
<div class="page-header">
    <h3 class="page-title">
        <span class="page-title-icon bg-gradient-primary text-white mr-2">
            <i class="mdi mdi-account-multiple"></i>
        </span> Manage Voters
    </h3>
<nav aria-label="breadcrumb">
        <ol class="breadcrumb">
            <li class="breadcrumb-item"><a href="../index.php">Dashboard</a></li>
            <li class="breadcrumb-item active" aria-current="page">Voters</li>
        </ol>
    </nav>
</div>

<div class="row">
    <div class="col-lg-12 grid-margin stretch-card">
        <div class="card">
            <div class="card-body">
                <h4 class="card-title">Voters</h4>
                <p class="card-description">
                    List of all voters for the election.
                </p>
                <?php echo $message; ?>
                <button type="button" class="btn btn-gradient-primary btn-sm mb-3" data-bs-toggle="modal" data-bs-target="#addVoterModal">
                    Add New Voter
                </button>

        <div class="table-responsive">
            <table class="table table-hover">
                <thead>
                <tr>
                    <th>ID</th>
                    <th>Staff ID</th>
                    <th>Name</th>
                    <th>Email</th>
                    <th>Department</th>
                    <th>Voted</th>
                    <th>Actions</th>
                </tr>
                </thead>
                <tbody>

                <?php if (empty($voters)): ?>
                    <tr><td colspan="6" class="text-center">No voters found.</td></tr>
                <?php else: ?>
                    <?php foreach ($voters as $v): ?>
                        <tr>
                            <td><?= $v["id"] ?></td>
                            <td><?= htmlspecialchars($v["staff_id"]) ?></td>
                            <td><?= htmlspecialchars($v["full_name"]) ?></td>
                            <td><?= htmlspecialchars($v["email"]) ?></td>
                            <td><?= htmlspecialchars($v["department"]) ?></td>
                            <td>
                                <label class="badge badge-<?= $v["has_voted"] ? 'success' : 'danger'; ?>">
                                    <?= $v["has_voted"] ? 'Yes' : 'No' ?>
                                </label>
                            </td>
                            <td>

                                <!-- EDIT -->
                                <button class="btn btn-info btn-sm edit-btn"
                                        data-id="<?= $v['id'] ?>"
                                        data-staff="<?= htmlspecialchars($v['staff_id']) ?>"
                                        data-name="<?= htmlspecialchars($v['full_name']) ?>"
                                        data-email="<?= htmlspecialchars($v['email']) ?>"
                                        data-dept="<?= htmlspecialchars($v['department']) ?>"
                                        data-bs-toggle="modal"
                                        data-bs-target="#editVoterModal">
                                    Edit
                                </button>

                                <!-- DELETE -->
                                <a href="?action=delete&id=<?= $v['id'] ?>"
                                   onclick="return confirm('Delete this voter?')"
                                   class="btn btn-danger btn-sm">
                                    Delete
                                </a>

                                <!-- CANCEL VOTE (Only when voter has voted) -->
                                <?php if ($v["has_voted"] == 1): ?>
                                    <a href="?action=cancel_vote&id=<?= $v['id'] ?>"
                                       onclick="return confirm('Cancel vote for this voter? They will be able to vote again.')"
                                       class="btn btn-warning btn-sm mt-1">
                                        Cancel Vote
                                    </a>
                                <?php endif; ?>

                            </td>

                        </tr>
                    <?php endforeach; ?>
                <?php endif; ?>

                </tbody>
            </table>
        </div>

    </div>
</div>

<!-- ADD VOTER MODAL (EXACTLY LIKE ORIGINAL FORM) -->
<div class="modal fade" id="addVoterModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">

            <div class="modal-header">
                <h5 class="modal-title"><i class="mdi mdi-account-plus"></i> Add Single Voter</h5>
                <button class="btn-close" data-bs-dismiss="modal"></button>
            </div>

            <form method="POST" action="">
                <div class="modal-body">

                    <div class="form-group mb-3">
                        <label>Staff ID (Username)</label>
                        <input type="text" class="form-control" name="staff_id" required>
                    </div>

                    <div class="form-group mb-3">
                        <label>Full Name</label>
                        <input type="text" class="form-control" name="full_name" required>
                    </div>

                    <div class="form-group">
                        <label>Email</label>
                        <input type="text" class="form-control" id="email" name="email" placeholder="Email" required>
                    </div>

                    <div class="form-group mb-3">
                        <label>Department</label>
                        <input type="text" class="form-control" name="department" required>
                    </div>

                    <div class="form-group mb-3">
                        <label>Password</label>
                        <input type="text" class="form-control" value="A unique password will be generated automatically." disabled>
                        <small class="text-muted">A unique password will be generated and emailed.</small>
                    </div>

                </div>

                <div class="modal-footer">
                    <button class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button class="btn btn-primary" name="add_voter">Add Voter</button>
                </div>

            </form>

        </div>
    </div>
</div>

<!-- EDIT VOTER MODAL -->
<div class="modal fade" id="editVoterModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">

            <div class="modal-header">
                <h5 class="modal-title">Edit Voter</h5>
                <button class="btn-close" data-bs-dismiss="modal"></button>
            </div>

            <form method="POST" action="">
                <div class="modal-body">

                    <input type="hidden" id="edit_voter_id" name="voter_id">

                    <div class="form-group mb-3">
                        <label>Staff ID</label>
                        <input type="text" class="form-control" id="edit_staff_id" name="staff_id" required>
                    </div>

                    <div class="form-group mb-3">
                        <label>Full Name</label>
                        <input type="text" class="form-control" id="edit_full_name" name="full_name" required>
                    </div>

                    <div class="form-group">
                        <label>Email</label>
                        <input type="text" class="form-control" id="email" name="email" placeholder="Email" required>
                    </div>

                    <div class="form-group mb-3">
                        <label>Department</label>
                        <input type="text" class="form-control" id="edit_department" name="department" required>
                    </div>

                </div>

                <div class="modal-footer">
                    <button class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                    <button class="btn btn-primary" name="edit_voter">Update</button>
                </div>
            </form>

        </div>
    </div>
</div>

<?php require_once '../../includes/footer.php'; ?>

<!-- JS FOR EDIT BUTTON -->
<script>
document.addEventListener("DOMContentLoaded", function () {
    document.querySelectorAll(".edit-btn").forEach(function (btn) {

        btn.addEventListener("click", function () {

            document.getElementById("edit_voter_id").value = this.dataset.id;
            document.getElementById("edit_staff_id").value = this.dataset.staff;
            document.getElementById("edit_full_name").value = this.dataset.name;
            document.getElementById("edit_department").value = this.dataset.dept;

        });
    });
});
</script>
