<?php
// Bulk Staff Upload
require_once '../../includes/header.php';
require_once '../../includes/email_mock.php';
check_admin_login();

$message = '';

// Voters now have unique passwords.
function generate_random_password($length = 10) {
    $chars = 'abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789!@#$%^&*()';
    $password = '';
    for ($i = 0; $i < $length; $i++) {
        $password .= $chars[rand(0, strlen($chars) - 1)];
    }
    return $password;
}

if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['bulk_upload'])) {
    if (isset($_FILES['csv_file']) && $_FILES['csv_file']['error'] == 0) {
        $file_tmp_path = $_FILES['csv_file']['tmp_name'];
        $file_type = $_FILES['csv_file']['type'];
        $file_size = $_FILES['csv_file']['size'];

        // Basic validation
        if ($file_type != 'text/csv' && $file_type != 'application/vnd.ms-excel') {
            $message = '<div class="alert alert-danger">Invalid file type. Please upload a CSV file.</div>';
        } else {
            $handle = fopen($file_tmp_path, "r");
            if ($handle !== FALSE) {
                $row_count = 0;
                $success_count = 0;
                $error_count = 0;
                $error_details = [];

                // Skip header row
                fgetcsv($handle);

                // Prepare the insert statement outside the loop for efficiency
                $sql = "INSERT INTO voters (staff_id, full_name, department, password) VALUES (?, ?, ?, ?)";
                $stmt = $conn->prepare($sql);
                $stmt->bind_param("ssss", $staff_id, $full_name, $department, $hashed_password);
                
                $passwords_generated = []; // To store generated passwords for mock email

                while (($data = fgetcsv($handle, 1000, ",")) !== FALSE) {
                    // Generate unique password and hash it
                    $raw_password = generate_random_password(10);
                    $hashed_password = password_hash($raw_password, PASSWORD_BCRYPT);
                    $passwords_generated[] = ['staff_id' => trim($data[0]), 'password' => $raw_password, 'full_name' => trim($data[1])];
                    $row_count++;
                    // Assuming CSV format: staff_id, full_name, department
                    if (count($data) >= 3) {
                        $staff_id = trim($data[0]);
                        $full_name = trim($data[1]);
                        $department = trim($data[2]);

                        if (empty($staff_id) || empty($full_name) || empty($department)) {
                            $error_details[] = "Row $row_count: Missing required data.";
                            $error_count++;
                            continue;
                        }

                        if ($stmt->execute()) {
                            $success_count++;
                        } else {
                            if ($conn->errno == 1062) {
                                $error_details[] = "Row $row_count: Staff ID '{$staff_id}' already exists.";
                            } else {
                                $error_details[] = "Row $row_count: Database error - " . $stmt->error;
                            }
                            $error_count++;
                        }
                    } else {
                        $error_details[] = "Row $row_count: Incorrect number of columns.";
                        $error_count++;
                    }
                }

                $stmt->close();
                fclose($handle);

                // Mock send emails for successful additions
                foreach ($passwords_generated as $voter_data) {
                    // Only send mock email if the voter was successfully added (not a duplicate error)
                    // This is a simplification, a real system would track successful inserts better
                    // For now, we assume all generated passwords correspond to an attempted insert.
                    send_voter_credentials_email($voter_data['staff_id'], $voter_data['password'], $voter_data['full_name']);
                }

                if ($error_count == 0) {
                    $message = '<div class="alert alert-success">Bulk upload successful! ' . $success_count . ' voters added. Mock emails logged to uploads/email_log.txt.</div>';
                } else {
                    $message = '<div class="alert alert-warning">Bulk upload completed with ' . $success_count . ' successful additions and ' . $error_count . ' errors. Mock emails logged. See details below.</div>';
                }
            } else {
                $message = '<div class="alert alert-danger">Error opening the uploaded file.</div>';
            }
        }
    } else {
        $message = '<div class="alert alert-danger">Please select a file to upload.</div>';
    }
}
?>

<div class="page-header">
    <h3 class="page-title">
        <span class="page-title-icon bg-gradient-primary text-white mr-2">
            <i class="mdi mdi-upload"></i>
        </span> Bulk Staff Upload
    </h3>
    <nav aria-label="breadcrumb">
        <ol class="breadcrumb">
            <li class="breadcrumb-item"><a href="../index.php">Dashboard</a></li>
            <li class="breadcrumb-item"><a href="view_voters.php">Manage Voters</a></li>
            <li class="breadcrumb-item active" aria-current="page">Bulk Upload</li>
        </ol>
    </nav>
</div>

<div class="row">
    <div class="col-md-8 grid-margin stretch-card">
        <div class="card">
            <div class="card-body">
                <h4 class="card-title">Upload Voter CSV/Excel File</h4>
                <p class="card-description">
                    Upload a CSV file to import multiple staff members (voters) at once.
                </p>
                <?php echo $message; ?>
                <div class="alert alert-info">
                    <strong>CSV Format Required:</strong> The file must be a comma-separated values (CSV) file with the following columns in order (no header row is expected, but if present, it will be skipped):
                    <ol>
                        <li><code>staff_id</code> (Unique username)</li>
                        <li><code>full_name</code></li>
                        <li><code>department</code></li>
                    </ol>
                    A unique password will be generated for each voter and a mock email with credentials will be logged to `uploads/email_log.txt`.
                </div>
                <form class="forms-sample" method="POST" action="<?php echo htmlspecialchars($_SERVER["PHP_SELF"]); ?>" enctype="multipart/form-data">
                    <div class="form-group">
                        <label for="csv_file">Select CSV File</label>
                        <input type="file" class="form-control-file" id="csv_file" name="csv_file" accept=".csv" required>
                    </div>
                    <button type="submit" name="bulk_upload" class="btn btn-gradient-primary mr-2">Upload and Import</button>
                </form>

                <?php if (!empty($error_details)): ?>
                    <h5 class="mt-4 text-danger">Import Errors:</h5>
                    <ul class="list-group">
                        <?php foreach ($error_details as $error): ?>
                            <li class="list-group-item list-group-item-danger"><?php echo htmlspecialchars($error); ?></li>
                        <?php endforeach; ?>
                    </ul>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>

<?php
require_once '../../includes/footer.php';
?>
