<?php
// View Admin Users Management
require_once '../../includes/header.php';
check_admin_login();

// Only super_admin can access this page
if ($_SESSION['admin_role'] !== 'super_admin') {
    header("location: ../index.php");
    exit;
}

$message = '';

/* -----------------------------
   DELETE USER
------------------------------ */
if (isset($_GET['action']) && $_GET['action'] === 'delete' && isset($_GET['id'])) {
    $id = intval($_GET['id']);

    // Prevent deleting own account
    if ($id == $_SESSION['admin_id']) {
        $message = '<div class="alert alert-danger">You cannot delete your own account.</div>';
    } else {
        $sql = "DELETE FROM users WHERE id = ?";
        if ($stmt = $conn->prepare($sql)) {
            $stmt->bind_param("i", $id);
            if ($stmt->execute()) {
                $message = '<div class="alert alert-success">Admin user deleted successfully.</div>';
            } else {
                $message = '<div class="alert alert-danger">Error: '.$conn->error.'</div>';
            }
            $stmt->close();
        }
    }
}

/* -----------------------------
   EDIT USER
------------------------------ */
if (isset($_POST['edit_user'])) {

    $id        = $_POST['user_id'];
    $name      = trim($_POST['full_name']);
    $role      = $_POST['role'];

    if (!empty($name) && !empty($role) && is_numeric($id)) {

        $sql = "UPDATE users SET full_name=?, role=? WHERE id=?";
        if ($stmt = $conn->prepare($sql)) {
            $stmt->bind_param("ssi", $name, $role, $id);

            if ($stmt->execute()) {
                $message = '<div class="alert alert-success">Admin user updated successfully.</div>';
            } else {
                $message = '<div class="alert alert-danger">Error: '.$conn->error.'</div>';
            }

            $stmt->close();
        }

    } else {
        $message = '<div class="alert alert-warning">Invalid update data.</div>';
    }
}

/* -----------------------------
   ADD NEW ADMIN USER (Modal)
------------------------------ */
if (isset($_POST["add_user"])) {

    $full_name = trim($_POST["full_name"]);
    $email     = trim($_POST["email"]);
    $role      = trim($_POST["role"]);
    $password  = trim($_POST["password"]);
    $hashed    = password_hash($password, PASSWORD_BCRYPT);

    if (!empty($full_name) && !empty($email) && !empty($password) && !empty($role)) {

        $sql = "INSERT INTO users (full_name, email, role, password) VALUES (?, ?, ?, ?)";

        if ($stmt = $conn->prepare($sql)) {
            $stmt->bind_param("ssss", $full_name, $email, $role, $hashed);

            if ($stmt->execute()) {
                $message = '<div class="alert alert-success">New admin user added successfully.</div>';
            } else {
                if ($conn->errno == 1062) {
                    $message = '<div class="alert alert-danger">Email already exists.</div>';
                } else {
                    $message = '<div class="alert alert-danger">Error: '.$conn->error.'</div>';
                }
            }
            $stmt->close();
        }

    } else {
        $message = '<div class="alert alert-danger">All fields are required.</div>';
    }
}

/* -----------------------------
   FETCH USERS
------------------------------ */
$users = [];
$sql = "SELECT id, full_name, email, role, created_at FROM users ORDER BY role DESC, full_name ASC";
$res = $conn->query($sql);
while ($row = $res->fetch_assoc()) {
    $users[] = $row;
}
?>

<!-- PAGE HEADER -->
<div class="page-header">
    <h3 class="page-title">
        <span class="page-title-icon bg-gradient-primary text-white mr-2">
            <i class="mdi mdi-account-key"></i>
        </span> Manage Admin Users
    </h3>
<nav aria-label="breadcrumb">
        <ol class="breadcrumb">
            <li class="breadcrumb-item"><a href="../index.php">Dashboard</a></li>
            <li class="breadcrumb-item active" aria-current="page">Users</li>
        </ol>
    </nav>
</div>

<div class="row">
    <div class="col-lg-12 grid-margin stretch-card">
        <div class="card">
            <div class="card-body">
                <h4 class="card-title">Users</h4>
                <p class="card-description">
                    List of all users.
                </p>
                <?php echo $message; ?>
                <button type="button" class="btn btn-gradient-primary btn-sm mb-3" data-bs-toggle="modal" data-bs-target="#addUserModal">
                    Add New Admin User
                </button>

        <div class="table-responsive">
            <table class="table table-hover">

                <thead>
                <tr>
                    <th>ID</th>
                    <th>Full Name</th>
                    <th>Email</th>
                    <th>Role</th>
                    <th>Actions</th>
                </tr>
                </thead>

                <tbody>
                <?php if (empty($users)): ?>
                    <tr><td colspan="5" class="text-center">No admin users found.</td></tr>
                <?php else: ?>
                    <?php foreach ($users as $u): ?>
                        <tr>
                            <td><?= $u['id'] ?></td>
                            <td><?= htmlspecialchars($u['full_name']) ?></td>
                            <td><?= htmlspecialchars($u['email']) ?></td>
                            <td>
                                <span class="badge badge-<?= $u['role']=='super_admin'?'danger':'info' ?>">
                                    <?= ucwords(str_replace('_',' ',$u['role'])) ?>
                                </span>
                            </td>
                            <td>

                                <!-- EDIT BUTTON -->
                                <button class="btn btn-info btn-sm edit-btn"
                                        data-id="<?= $u['id'] ?>"
                                        data-name="<?= htmlspecialchars($u['full_name']) ?>"
                                        data-email="<?= htmlspecialchars($u['email']) ?>"
                                        data-role="<?= $u['role'] ?>"
                                        data-bs-toggle="modal"
                                        data-bs-target="#editUserModal">
                                    Edit
                                </button>

                                <!-- DELETE BUTTON -->
                                <?php if ($u['id'] != $_SESSION['admin_id']): ?>
                                    <a href="?action=delete&id=<?= $u['id'] ?>"
                                       onclick="return confirm('Delete this admin user?')"
                                       class="btn btn-danger btn-sm">Delete</a>
                                <?php endif; ?>

                            </td>
                        </tr>
                    <?php endforeach; ?>
                <?php endif; ?>
                </tbody>

            </table>
        </div>
    </div>
</div>

<!-- ADD USER MODAL -->
<div class="modal fade" id="addUserModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">

            <div class="modal-header">
                <h5 class="modal-title"><i class="mdi mdi-account-plus"></i> Add Admin User</h5>
                <button class="btn-close" data-bs-dismiss="modal"></button>
            </div>

            <form method="POST" action="">
                <div class="modal-body">

                    <div class="form-group mb-3">
                        <label>Full Name</label>
                        <input type="text" name="full_name" class="form-control" required>
                    </div>

                    <div class="form-group mb-3">
                        <label>Email (must be unique)</label>
                        <input type="email" name="email" class="form-control" required>
                    </div>

                    <div class="form-group mb-3">
                        <label>Password</label>
                        <input type="text" name="password" class="form-control" required>
                        <small class="text-muted">Password will be encrypted automatically.</small>
                    </div>

                    <div class="form-group mb-3">
                        <label>Role</label>
                        <select name="role" class="form-control" required>
                            <option value="normal_admin">Normal Admin</option>
                            <option value="super_admin">Super Admin</option>
                        </select>
                    </div>

                </div>

                <div class="modal-footer">
                    <button class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button class="btn btn-primary" name="add_user">Add Admin</button>
                </div>
            </form>

        </div>
    </div>
</div>

<!-- EDIT USER MODAL -->
<div class="modal fade" id="editUserModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">

            <div class="modal-header">
                <h5 class="modal-title">Edit Admin User</h5>
                <button class="btn-close" data-bs-dismiss="modal"></button>
            </div>

            <form method="POST" action="">
                <div class="modal-body">

                    <input type="hidden" id="edit_user_id" name="user_id">

                    <div class="form-group mb-3">
                        <label>Full Name</label>
                        <input type="text" id="edit_full_name" name="full_name" class="form-control" required>
                    </div>

                    <div class="form-group mb-3">
                        <label>Role</label>
                        <select id="edit_role" name="role" class="form-control" required>
                            <option value="normal_admin">Normal Admin</option>
                            <option value="super_admin">Super Admin</option>
                        </select>
                    </div>

                    <small class="text-muted">Passwords cannot be changed here.</small>

                </div>

                <div class="modal-footer">
                    <button class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                    <button class="btn btn-primary" name="edit_user">Update User</button>
                </div>

            </form>

        </div>
    </div>
</div>

<?php require_once '../../includes/footer.php'; ?>

<!-- JS: Populate Edit Modal -->
<script>
document.addEventListener("DOMContentLoaded", function () {

    document.querySelectorAll(".edit-btn").forEach(btn => {

        btn.addEventListener("click", function () {

            document.getElementById("edit_user_id").value   = this.dataset.id;
            document.getElementById("edit_full_name").value = this.dataset.name;
            document.getElementById("edit_role").value      = this.dataset.role;

        });

    });

});
</script>
