<?php
// Election Results Module
require_once '../../includes/header.php';
check_admin_login();

// --- Fetch Results Data ---

// 1. Total Votes per Candidate
$candidate_votes = [];
$sql_candidate_votes = "SELECT c.candidate_name, p.position_name, COUNT(v.id) AS vote_count
                        FROM candidates c
                        LEFT JOIN votes v ON c.id = v.candidate_id
                        JOIN positions p ON c.position_id = p.id
                        GROUP BY c.id, c.candidate_name, p.position_name
                        ORDER BY p.position_name ASC, vote_count DESC";
if ($result = $conn->query($sql_candidate_votes)) {
    while ($row = $result->fetch_assoc()) {
        $candidate_votes[] = $row;
    }
    $result->free();
}

// 2. Vote Breakdown by Position and Determine Winners
$position_results = [];
$chart_data = [];
$current_position = '';
$max_votes = -1;
$winner_name = '';

foreach ($candidate_votes as $vote) {
    $position = $vote['position_name'];
    $candidate = $vote['candidate_name'];
    $count = $vote['vote_count'];

    if ($position !== $current_position) {
        // Finalize previous position's winner
        if ($current_position !== '') {
            $position_results[$current_position]['winner'] = $winner_name;
        }

        // Start new position
        $current_position = $position;
        $max_votes = -1;
        $winner_name = '';
        $position_results[$position] = [
            'candidates' => [],
            'winner' => ''
        ];
        $chart_data[$position] = [
            'labels' => [],
            'data' => []
        ];
    }

    // Update current position's data
    $position_results[$position]['candidates'][] = [
        'name' => $candidate,
        'votes' => $count
    ];
    $chart_data[$position]['labels'][] = $candidate;
    $chart_data[$position]['data'][] = $count;

    // Determine winner
    if ($count > $max_votes) {
        $max_votes = $count;
        $winner_name = $candidate;
    } elseif ($count == $max_votes) {
        $winner_name .= " (Tie with {$candidate})";
    }
}

// Finalize the last position's winner
if ($current_position !== '') {
    $position_results[$current_position]['winner'] = $winner_name;
}

// --- CSV Download Logic ---
if (isset($_GET['action']) && $_GET['action'] == 'download_csv') {
    header('Content-Type: text/csv; charset=utf-8');
    header('Content-Disposition: attachment; filename=election_results_' . date('Ymd') . '.csv');
    $output = fopen('php://output', 'w');

    // CSV Header
    fputcsv($output, ['Position', 'Candidate', 'Votes', 'Is Winner']);

    foreach ($position_results as $position_name => $data) {
        $winner_info = $data['winner'];
        foreach ($data['candidates'] as $candidate) {
            $is_winner = (strpos($winner_info, $candidate['name']) !== false) ? 'Yes' : 'No';
            fputcsv($output, [
                $position_name,
                $candidate['name'],
                $candidate['votes'],
                $is_winner
            ]);
        }
    }

    fclose($output);
    exit;
}

?>

<div class="page-header">
    <h3 class="page-title">
        <span class="page-title-icon bg-gradient-primary text-white mr-2">
            <i class="mdi mdi-chart-bar"></i>
        </span> Election Results
    </h3>
    <nav aria-label="breadcrumb">
        <ol class="breadcrumb">
            <li class="breadcrumb-item"><a href="../index.php">Dashboard</a></li>
            <li class="breadcrumb-item active" aria-current="page">Results</li>
        </ol>
    </nav>
</div>

<div class="row">
    <div class="col-12 grid-margin">
        <a href="?action=download_csv" class="btn btn-gradient-success btn-sm mb-3">
            <i class="mdi mdi-download btn-icon-prepend"></i> Download Results CSV
        </a>
    </div>
</div>

<?php if (empty($position_results)): ?>
    <div class="alert alert-info">No positions or votes recorded yet.</div>
<?php else: ?>
    <?php foreach ($position_results as $position_name => $data): ?>
        <div class="row">
            <div class="col-lg-12 grid-margin stretch-card">
                <div class="card">
                    <div class="card-body">
                        <h4 class="card-title"><?php echo htmlspecialchars($position_name); ?> Results</h4>
                        <div class="row">
                            <div class="col-md-4">
                                <div class="card bg-gradient-info card-img-holder text-white">
                                    <div class="card-body">
                                        <img src="../../assets/images/dashboard/circle.svg" class="card-img-absolute" alt="circle-image" />
                                        <h4 class="font-weight-normal mb-3">Winner <i class="mdi mdi-trophy mdi-24px float-right"></i></h4>
                                        <h2 class="mb-5"><?php echo htmlspecialchars($data['winner']); ?></h2>
                                        <h6 class="card-text">Total Candidates: <?php echo count($data['candidates']); ?></h6>
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-8">
                                <canvas id="chart-<?php echo str_replace(' ', '-', $position_name); ?>" style="height:250px"></canvas>
                            </div>
                        </div>
                        <div class="table-responsive mt-4">
                            <table class="table table-bordered">
                                <thead>
                                    <tr>
                                        <th>Candidate</th>
                                        <th>Total Votes</th>
                                        <th>Status</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($data['candidates'] as $candidate): ?>
                                        <tr>
                                            <td><?php echo htmlspecialchars($candidate['name']); ?></td>
                                            <td><?php echo $candidate['votes']; ?></td>
                                            <td>
                                                <?php if (strpos($data['winner'], $candidate['name']) !== false): ?>
                                                    <label class="badge badge-success">Winner</label>
                                                <?php else: ?>
                                                    <label class="badge badge-secondary">Runner-up</label>
                                                <?php endif; ?>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    <?php endforeach; ?>
<?php endif; ?>

<script>
    document.addEventListener('DOMContentLoaded', function() {
        var chartData = <?php echo json_encode($chart_data); ?>;
        var backgroundColors = [
            'rgba(255, 99, 132, 0.5)',
            'rgba(54, 162, 235, 0.5)',
            'rgba(255, 206, 86, 0.5)',
            'rgba(75, 192, 192, 0.5)',
            'rgba(153, 102, 255, 0.5)',
            'rgba(255, 159, 64, 0.5)'
        ];
        var borderColors = [
            'rgba(255, 99, 132, 1)',
            'rgba(54, 162, 235, 1)',
            'rgba(255, 206, 86, 1)',
            'rgba(75, 192, 192, 1)',
            'rgba(153, 102, 255, 1)',
            'rgba(255, 159, 64, 1)'
        ];

        for (var position in chartData) {
            if (chartData.hasOwnProperty(position)) {
                var positionKey = position.replace(/ /g, '-');
                var ctx = document.getElementById('chart-' + positionKey).getContext('2d');
                
                var data = chartData[position];
                var colors = data.labels.map((_, i) => backgroundColors[i % backgroundColors.length]);
                var borders = data.labels.map((_, i) => borderColors[i % borderColors.length]);

                new Chart(ctx, {
                    type: 'bar',
                    data: {
                        labels: data.labels,
                        datasets: [{
                            label: 'Total Votes',
                            data: data.data,
                            backgroundColor: colors,
                            borderColor: borders,
                            borderWidth: 1
                        }]
                    },
                    options: {
                        responsive: true,
                        maintainAspectRatio: false,
                        scales: {
                            y: {
                                beginAtZero: true,
                                ticks: {
                                    stepSize: 1
                                }
                            }
                        },
                        plugins: {
                            legend: {
                                display: false
                            }
                        }
                    }
                });
            }
        }
    });
</script>

<?php
require_once '../../includes/footer.php';
?>
